import copy
import os

import pytest
from pytestqt.qtbot import QtBot
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import QFileDialog, QMessageBox

from grm.dialogs.LandCover_dialog import LandCoverDialog
from grm.lib.Util import MsTitle

# Set environment variables (PyQt 실행용)
if os.environ.get("QT_QPA_PLATFORM_PLUGIN_PATH") is None:
    os.environ[
        "QT_QPA_PLATFORM_PLUGIN_PATH"
    ] = r"C:\Program Files\QGIS 3.10\apps\Qt5\plugins"
if os.environ.get("PROJ_LIB") is None:
    os.environ["PROJ_LIB"] = "C:\\PROGRA~1\\QGIS3~1.10\\share\\proj"


class TestLandCoverDialog:
    """
    LandCoverDialog 테스트
    """

    __xmltodict = {
        "GRMProject": {
            "ProjectSettings": {
                "LandCoverDataType": "File",
                "SoilTextureDataType": "File",
                "SoilDepthDataType": "File",
                "LandCoverVATFile": "",
                "LandCoverFile": "",
                "SoilTextureVATFile": "",
                "SoilDepthVATFile": "",
                "SoilTextureFile": "",
                "SoilDepthFile": "",
                "LAIFile": "",
                "BlaneyCriddleCoefDataFile": "",
            }
        }
    }

    @classmethod
    def setup_class(self):
        MsTitle("ClimateDataDialog")

    @pytest.fixture(scope="class")
    def monkeyclass(self) -> pytest.MonkeyPatch:
        """
        Mocking QMessageBox
        """
        from _pytest.monkeypatch import MonkeyPatch

        mpatch = MonkeyPatch()
        mpatch.setattr(QMessageBox, "information", lambda *args: "information")
        mpatch.setattr(QMessageBox, "warning", lambda *args: "warning")
        mpatch.setattr(
            QFileDialog,
            "getOpenFileName",
            lambda *args: ("/path/mock/file/test.txt", "txt"),
        )
        yield mpatch
        mpatch.undo()

    @property
    def xmltodict(self):
        return copy.deepcopy(self.__xmltodict)

    def __set_init_with_no_error(self, qtbot: QtBot, dialog: LandCoverDialog):
        qtbot.mouseClick(dialog.rbtUseConstLCAtt, Qt.LeftButton)
        qtbot.mouseClick(dialog.rbtUseConstTextureAtt, Qt.LeftButton)
        qtbot.mouseClick(dialog.rbtUseConstDepth, Qt.LeftButton)
        qtbot.keyClicks(dialog.txtCoefficient, "1")
        qtbot.keyClicks(dialog.txtImpervious, "1")
        qtbot.keyClicks(dialog.txtPorosity, "1")
        qtbot.keyClicks(dialog.txtEffective_porosity, "1")
        qtbot.keyClicks(dialog.txtSuction_head, "1")
        qtbot.keyClicks(dialog.txtConductiovity, "1")
        qtbot.keyClicks(dialog.txtSoil_depth, "1")

    def test_default(self, qtbot: QtBot):
        assert LandCoverDialog(self.xmltodict) is not None

    def test_keyerror(self, qtbot: QtBot):
        with pytest.raises(KeyError):
            LandCoverDialog()

    def test_LAIFile_with_no_require_option(
        self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch
    ):
        xml = self.xmltodict
        is_info = False

        def mock_warning():
            nonlocal is_info
            is_info = True

        monkeyclass.setattr(QMessageBox, "information", lambda *args: mock_warning())
        dialog = LandCoverDialog(xml)
        qtbot.addWidget(dialog)
        qtbot.mouseClick(dialog.btn_LAI_file, Qt.LeftButton)
        qtbot.mouseClick(dialog.btnOK, Qt.LeftButton)
        assert is_info
        assert xml == self.xmltodict
        assert dialog.path_LAI_file.text() is not ""

    def test_LAIFile(self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch):
        xml = self.xmltodict
        is_error = False

        def mock_error():
            nonlocal is_error
            is_error = True

        monkeyclass.setattr(QMessageBox, "warning", lambda *args: mock_error())

        dialog = LandCoverDialog(xml)
        qtbot.addWidget(dialog)
        self.__set_init_with_no_error(qtbot, dialog)
        qtbot.mouseClick(dialog.btn_LAI_file, Qt.LeftButton)
        qtbot.mouseClick(dialog.btnOK, Qt.LeftButton)
        assert (
            xml["GRMProject"]["ProjectSettings"]["LAIFile"]
            == "/path/mock/file/test.txt"
        )
        assert dialog.path_LAI_file.text() == "/path/mock/file/test.txt"
        assert not is_error

    def test_blaney_criddle_coef_with_no_require_option(
        self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch
    ):
        xml = self.xmltodict
        is_info = False

        def mock_warning():
            nonlocal is_info
            is_info = True

        monkeyclass.setattr(QMessageBox, "information", lambda *args: mock_warning())
        dialog = LandCoverDialog(xml)
        qtbot.addWidget(dialog)
        qtbot.mouseClick(dialog.btn_bianey_criddle_coet, Qt.LeftButton)
        qtbot.mouseClick(dialog.btnOK, Qt.LeftButton)
        assert is_info
        assert xml == self.xmltodict
        assert dialog.path_blaney_criddle_coet.text() is not ""

    def test_blaney_criddle_coef(self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch):
        xml = self.xmltodict
        is_error = False

        def mock_error():
            nonlocal is_error
            is_error = True

        monkeyclass.setattr(QMessageBox, "warning", lambda *args: mock_error())

        dialog = LandCoverDialog(xml)
        qtbot.addWidget(dialog)
        self.__set_init_with_no_error(qtbot, dialog)
        qtbot.mouseClick(dialog.btn_bianey_criddle_coet, Qt.LeftButton)
        qtbot.mouseClick(dialog.btnOK, Qt.LeftButton)
        assert (
            xml["GRMProject"]["ProjectSettings"]["BlaneyCriddleCoefDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert dialog.path_blaney_criddle_coet.text() == "/path/mock/file/test.txt"
        assert not is_error
